#!/bin/bash

# Input files are copied to the output with textual substitutions.  
#	-o outfile	This file receives the output; only makes sense with
#			just one input file.
# Lacking -o, the output has the same name as the input minus one extension,
# e.g. dir/name.html.m4 would be copied into dir/name.html .

# Files on the command line are processed.  If none are given, then 
# each file in the same directory as this script with an extension of .m4
# is processed.  

# Available substitutions:
#	%h	One-component name of host, from uname -n
#	%H	One-component name of host, capitalized
#	%f	FQDN of host, from DNS, normally lower case
#	%d	One-component realm name from FQDN.  This is the name component
#		after the hostname.
#	%D	One-component realm name, all caps
#	%r	Complete realm, i.e. FQDN minus the hostname
#	%u	Pre-filled user name, from the autologin-user parameter in 
#		lightdm.conf (lightdm seems to not pass in this value).
#		Default is "" if not specified or this user has no homedir 
#		on this host.
#	%t	Timeout before auto-logging in, from autologin-user-timeout
#		(in seconds).  Default is 0.  
#	%S	Default session type, from autologin-session, which is marked
#		as unimplemented in the file.  

shopt -s nullglob

# Save the output file, if any.
if [ "X$1" = "X-o" ] ; then 
    opt_o=$2
    shift
    shift
fi

# Identify the input file(s).  
if [ $# -gt 0 ] ; then
    files=($*)
else
    case $0 in
	*/* )	dir=${0%/*} ; ;;
	* )	dir=l ; ;;
    esac
    files=(`eval echo $dir/*.m4`)
    if [ ${#files[*]} -le 0 ] ; then
	echo "No files on command line; no $dir/*.m4 either.  Exiting." 1>&2
	exit 4
    fi
fi

# Set up the textual substitutions.  
h=`uname -n`
H=${h^}
f=`hostname -d --fqdn`
r=${f#*.}
d=${r%%.*}
D=${d^^}

# Extracting variables from /etc/lightdm/lightdm.conf .  
# autologin_user = loginID to auto login, default "", only if it has a homedir.
# autologin_user_timeout = seconds to wait before auto login; if someone 
#	changes the value (by even 1 letter) the auto login is cancelled. 
#	Default is 0 for immediate login.  
# autologin_user_session = session to use.  

autologin_user_timeout=0
eval "` egrep '^ *autologin-(user|user-timeout|session)=' /etc/lightdm/lightdm.conf | sed -e 's/\([^=]\)-/\1_/g' `"
if [ -z "$autologin_user" ] || eval "[ ! -d ~$autologin_user ]" ; then autologin_user="" ; fi

# Process the files.  
for file in ${files[*]} ; do
    if [ -n "$opt_o" ] ; then 
	out=$opt_o 
    else
	out=${file%.*}
	case $out in
	    $file | "" | */ )	
			out=$file.new
			echo "Can't remove extension from $file; output to $out" 1>&2
			;;
	esac
    fi
    sed	-e "s/%h/$h/g" \
	-e "s/%H/$H/g" \
	-e "s/%f/$f/g" \
	-e "s/%r/$r/g" \
	-e "s/%d/$d/g" \
	-e "s/%D/$D/g" \
	-e "s/%u/$autologin_user/g" \
	-e "s/%t/$autologin_user_timeout/g" \
	-e "s/%S/$autologin_session/g" \
	$file > $out.new
    if [ $? -ne 0 ] ; then
	rm $out.new
    elif [ -e $out ] && cmp -s $out.new $out ; then
	rm $out.new
    else
	mv $out.new $out
    fi
done
